Scriptname RF:TravelHandlerQuest Extends Quest

Actor PlayerRef
SpaceshipReference MyShip

RF:FuelHandlerQuest Property FuelManager Auto Mandatory Const
RF:TutorialQuest Property TutorialManager Mandatory Const Auto
SQ_PlayerShipScript property SQ_PlayerShip Auto Const

Group Globals 
    GlobalVariable Property _RF_DistanceRestriction Mandatory Const Auto
    { Severity tied to SPEL record. 0:None 1:40% 2:60% 3:90% (should be functionally inter-system only) }
    GlobalVariable Property _RF_GravDebuff Mandatory Const Auto
    { 1: 2x 2: 4x 3:8x }
    GlobalVariable Property _RF_TRV_IsUsingEmergencyJump Mandatory Const Auto
    { This should be set in CheckRestrictions() to trigger our coinflip to damage stuff }
    GlobalVariable Property _RF_Sys_TravelLimitHardcore Mandatory Const Auto
    { Determines if the Limiter system runs or not where 1 is enabled }
    GlobalVariable Property _RF_Sys_AllowTravel Mandatory Const Auto
    { Travel system setting - 1 is enabled! }
    GlobalVariable Property _RF_Sys_Cooldowns Mandatory Const Auto
    { Cooldown system where 1 is enabled}
    GlobalVariable Property _RF_OverheatDuration Mandatory Const Auto
    { We reset this on end since it is set before firing based on event severity. }
    GlobalVariable Property _RF_CooldownDuration Mandatory Const Auto
EndGroup

Group Limiter
    MiscObject Property _RF_TravelKill Auto Mandatory Const
EndGroup

Group Messages
    Message Property _RF_Alert_GravJumpDisabled Mandatory Const Auto
    Message Property _RF_Alert_ShipDamage_Overheat Mandatory Const Auto
    Message Property _RF_Alert_GravJumpEnabled Mandatory Const Auto
    Message Property _RF_Alert_Map_ShipDamage Mandatory Const Auto
    Message Property _RF_Alert_ShipDamage Mandatory Const Auto
    Message Property _RF_Alert_TooManyJumps Mandatory Const Auto
EndGroup

Group Conditions
    Keyword Property LocTypeSENotAllowed Mandatory Const Auto
    ConditionForm Property _RF_InCooldownState Mandatory Const Auto
    { Indicates we are cooling down from a jump }
    ConditionForm Property _RF_Overheated Mandatory Const Auto
    { Indicates Overheat spell is still running - prevent re-enable from other modules }
EndGroup

Group Spells
    Spell Property _RF_GravDriveCooldownSpell Mandatory Const Auto
    { This fires if we jump too many times sequentially to disable far travel entirely }
    Spell Property _RF_GravDriveOverheatSpell Mandatory Const Auto
    { This fires if we encounter a disastrous scenario and disables grav jumping }
    Spell Property _rf_GravDriveDetectiveSpell Mandatory Const Auto
    { This is our counter to determine if we jumped recently }
EndGroup

Group ActorValues
    ActorValue Property SpaceshipGravDriveHealth Mandatory Const Auto
    ActorValue Property spaceshipgravjumpcalculation Mandatory Const Auto
EndGroup


bool GravJumpOK = true ; This is set by overheat or damage checks
bool FarTravelOK = true ; This is set by catastrophic events we are not using yet

;Counts for breakpoints
int BreakHigh = 70
int BreakNorm = 40
int BreakMerg = 25
int BreakCrit = 10

int LastPercent = 0

;Used for exceptions or out-of-fuel scenarios
bool UsingEmergencyJump = false ; Configurable... somewhere
bool WasGravJumpDisabled = false ; Updated in Restrictions() to avoid message spam

int SequentialFastJumps = 0 ; Set in HandleCooldowns

InputEnableLayer FarTravelInputLayerTHQ
InputEnableLayer GravJumpInputLayerTHQ

;System
bool Running = false ; Is the mod running
bool Hardcore = false ; This is our contraband limiter
bool Cooldown = false ; This is our cooldown system

; DEBUG - DO NOT TOUCH
bool DEBUGON = true ; Debug handler
bool DEBUGNOTIFY = false ; I can't tell how fast the scripts are running with this spamming me lol
bool DEBUGFORCE = false ; This makes coinflips always return true. DISABLE FOR PROD



;----------------------------------------------------
;----------------- SETTINGS HANDLER ----------------- 
;----------------------------------------------------


Function HandleSystemStartup()
    Running = True
    Hardcore = true
    Cooldown = true
    MyShip = FuelManager.GetShip("Travel")
    SmartLimitHandler(FuelManager.GetSysType(MyShip.GetCurrentLocation()))
EndFunction

; True indicates all elements of Travel are running
Function HandleSettings()
    MyShip = FuelManager.GetShip("Travel")
    int Mode = _RF_Sys_AllowTravel.GetValue() as int
    int HC = _RF_Sys_TravelLimitHardcore.GetValue() as int
    int CD = _RF_Sys_Cooldowns.GetValue() as int
    Hardcore = HC as bool
    Cooldown = CD as bool
    If Mode == 1 ; Mod enabled
        If !Running
            QuickTrace("Travel Manager starting from Standby")
        EndIf
        Running = True
    Else
        If Running
            QuickTrace("Travel Manager shutting down from Running")
        EndIf
        Running = False
    EndIf
    If Running
        CheckRestrictions()
        SmartLimitHandler(FuelManager.GetSysType(MyShip.GetCurrentLocation()))
    Else
        KillAllTravel()
        Hardcore = False
        Cooldown = False
    EndIf
EndFunction

Function KillAllTravel()
    QuickTrace("Running KillAll")
    DisableFar(false)
    DisableGrav(false)
    SequentialFastJumps = 0
    GravJumpOK = true 
    FarTravelOK = true
    UsingEmergencyJump = False
    WasGravJumpDisabled = False
    PlayerRef.DispelSpell(_RF_GravDriveOverheatSpell)
    PlayerRef.DispelSpell(_RF_GravDriveCooldownSpell)
    PlayerRef.DispelSpell(_rf_GravDriveDetectiveSpell)
    Limit(0)
EndFunction



;----------------------------------------------------
;----------------- DEBUG AND SYSTEM ----------------- 
;----------------------------------------------------



;Generic debug function that can be disabled for prod
Function QuickTrace(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Trace("RF TravelHandler: " + asTextToPrint)
        IF DEBUGNOTIFY ; Can log in realtime to avoid alt-tabbing constantly
            Debug.Notification("RTRAV: " + asTextToPrint)
        EndIF
    EndIF
EndFunction

Function QuickTraceVB(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Notification("RTRAV: " + asTextToPrint)
    EndIF
EndFunction

; Reduce an AV by X% of its current amount to avoid going negative if wanted
Function DamageValueSafe(ActorValue ValueToDamage, float afAmountToDamage = 0.75)
    float BaseAV = MyShip.GetBaseValue(ValueToDamage)
    float CurrAV = MyShip.GetValue(ValueToDamage)
    float CurrCent = CurrAV / BaseAV
    QuickTrace("DamageValueSafe got base " + BaseAV + " current " + CurrAV + " ratio " + CurrCent)
    float DamageToDo = CurrAV * afAmountToDamage
    MyShip.DamageValue(ValueToDamage, DamageToDo)
    QuickTrace("DamageValueSafe damaged" + ValueToDamage + " by " + DamageToDo + " resulting " + MyShip.GetValue(ValueToDamage) ) 
EndFunction

bool Function TravelSystemRunning()
    Return Running
EndFunction

bool Function GetOverheated()
    Return _RF_Overheated.IsTrue(PlayerRef)
EndFunction

bool Function GetCoolingDown()
    Return _RF_InCooldownState.IsTrue(PlayerRef)
EndFunction

bool Function CheckTravelStatus()
    Return GravJumpOK && FarTravelOK ; If either of these is false we skip notifs
EndFunction

int Function GFLI()
    int ful = FuelManager.GetFuelLevelInt()
    QuickTrace("GFLI: " + ful)
    Return ful
EndFunction


;----------------------------------------------------
;----------------- LIMITERS, LAYERS ----------------- 
;----------------------------------------------------



; This simply shuts off fast travel back to hubs. 
; Obviously it allows travel back to non-settled land locations, but whatever
Function SmartLimitHandler(int aiType)
    If Hardcore
        Location PlayerIsIn = MyShip.GetCurrentLocation()
        If RealFuel.IsSystemSettled(PlayerIsIn)
            Limit(false)
            QuickTrace("SmartLimitHandler cleared in Settled System")
        Else
            IF aiType > 1
                Limit()
                QuickTrace("SmartLimitHandler active with Type " + aiType )
            EndIf
        EndIf
    Else
        QuickTrace("Limits disabled")
    EndIf
EndFunction

Function Limit(bool abLimit = true)
    PlayerRef = Game.GetPlayer() ; I think this was just firing from another script before we got our player assigned
    If !abLimit
        PlayerRef.RemoveItem(_RF_TravelKill, 999, true)
    Else
        PlayerRef.AddItem(_RF_TravelKill, 1, true)
    EndIF
    int TK = PlayerRef.GetItemCount(_RF_TravelKill)
    QuickTrace("Limit fired " + abLimit + " and player now has " + TK + " limiters")
EndFunction

; True = No Grav Jumping
; False = Grav Jumping Is Enabled
Function DisableGrav(bool abDisabling)
    If abDisabling
        If (GravJumpInputLayerTHQ == None)
            GravJumpInputLayerTHQ = InputEnableLayer.Create()
            GravJumpInputLayerTHQ.EnableGravJump(False)
            GravJumpOK = False
            QuickTrace("Disabling Grav Jump")
        Else
            QuickTrace("EXCEPTION - DisableGrav called " + abDisabling +  " WITHOUT active input layer present!")
        EndIF
    Else
        If (GravJumpInputLayerTHQ != None)
            GravJumpInputLayerTHQ.EnableGravJump(True)
            GravJumpInputLayerTHQ.Delete()
            GravJumpInputLayerTHQ = None
            GravJumpOK = True
            QuickTrace("Enabling Grav Jump")
        Else
            QuickTrace("EXCEPTION - DisableGrav called " + abDisabling +  " WITH active input layer present!")
        EndIf
    EndIF
EndFunction

; True = No Far Travel
; False = Far Travel Enabled
Function DisableFar(bool abDisabling)
    If abDisabling
        If (FarTravelInputLayerTHQ == None)
            FarTravelInputLayerTHQ = InputEnableLayer.Create()
            FarTravelInputLayerTHQ.EnableFarTravel(False)
            FarTravelOK = False
            QuickTrace("Disabling Far Travel")
        Else
            QuickTrace("EXCEPTION - DisableFar called " + abDisabling +  " WITHOUT active input layer present!")
        EndIf
    Else
        If (FarTravelInputLayerTHQ != None)
            FarTravelInputLayerTHQ.EnableFarTravel(True)
            FarTravelInputLayerTHQ.Delete()
            FarTravelInputLayerTHQ = None
            FarTravelOK = True
            QuickTrace("Enabling Far Travel")
        Else
            QuickTrace("EXCEPTION - DisableFar called " + abDisabling +  " WITH active input layer present!")
        EndIf
    EndIF
EndFunction

Function EnableAndNotify()
    If (GravJumpInputLayerTHQ != None)
        If !GetOverheated()
            DisableGrav(false)
            If WasGravJumpDisabled
                TutorialManager.HelpMessage(_RF_Alert_GravJumpEnabled)
            EndIf
            WasGravJumpDisabled = False
        Else
            Debug.Notification("Grav Drive overheated! Repair to continue travel.")
            QuickTrace("EnableAndNotify called while overheated!!!")
        EndIF
    EndIF
EndFunction

Function DisableAndNotify(bool abNotify = true)
    bool Notify = false
    If (GravJumpInputLayerTHQ == None)
        DisableGrav(true)
        WasGravJumpDisabled = True
        TutorialManager.Nag(0)
    EndIF
EndFunction


;-------------------------------------------------
;---------------- TRAVEL HANDLERS ----------------
;-------------------------------------------------


; This is the entire core of our travel system disabler.
; It operates purely on fuel percentage to limit mobility.
Function CheckRestrictions()
    If Running
        UsingEmergencyJump = False
        bool TutorialTriggered = false
        Bool Disable = False
        int FuelLevel = GFLI()
        int Restriction = 0
        float Mult = 1.0 ; Skill scaling TBI
        If FuelLevel > ( BreakHigh * Mult)
            Restriction = 0
        ElseIf FuelLevel > ( BreakNorm * Mult )
            Restriction = 1
        ElseIf FuelLevel > ( BreakMerg * Mult )
            Restriction = 2
        ElseIf FuelLevel > BreakCrit
            Restriction = 3
            TutorialTriggered = TutorialManager.ShowTutorial(1)
        Elseif FuelLevel <= BreakCrit ; This disables mobility, the rest do little to nothing currently
            Restriction = 3
            QuickTrace("Restriction 3 on Critical")
            UsingEmergencyJump = True
            Disable = true
            TutorialTriggered = TutorialManager.ShowTutorial(2)
        EndIf
        If Disable
            If !TutorialTriggered ; I really hate message spam.
                DisableAndNotify() ; This just keeps the system alive and ticks up the tutorial counter
            Else
                DisableGrav(true)
            EndIF
        Else
            EnableAndNotify()
        EndIf
        _RF_DistanceRestriction.SetValue(0) ; This has been cut until further notice
        _RF_GravDebuff.SetValue(Restriction) ; This hampers our Grav Drive spool up speed. Conveniently, it uses the same values
        QuickTrace("CheckRestrictions: Tier" + Restriction + " from " + FuelLevel + "% fuel.")
        QuickTrace("Player has " + PlayerRef.GetValue(spaceshipgravjumpcalculation) + " gravjumpcalc value")
    Else
        int Restriction = 0
        _RF_DistanceRestriction.SetValue(0)
        _RF_GravDebuff.SetValue(0) 
        UsingEmergencyJump = False
        DisableGrav(false)
        DisableFar(false)
    EndIF
EndFunction

; If this fires we want to skip all fueling logic except to blow out what's left of our tanks. 
; We should also not fire ANY notif except this one - spell now has no notif on start for versatility
bool Function HandleEmergencyJump()
    QuickTrace("HandleEmergencyJump firing with UsingEmergency " + UsingEmergencyJump)
    bool DidDamageOccur = false ; If this happens we force drain fuel as well.
    If UsingEmergencyJump == true && GFLI() < 2 ; || _RF_TRV_IsUsingEmergencyJump.GetValue() == 1 ; Global is PURELY FOR DEBUG!
        ActorValue Hull = Game.GetHealthAV()
        int SaveRoll = Utility.RandomInt(0, 100)
        If SaveRoll <= 15
            QuickTrace("Failed coinflip with + " + Saveroll)
            DidDamageOccur = true
        EndIF
        If DidDamageOccur
            float Damage = 0.6 * FuelManager.GetVariance(0.15)
            QuickTrace("HandleEmergencyJump casting Grav Drive Overheat and damaging Hull for " + Damage * 100 as int + "%")
            Utility.Wait(0.1)
            TutorialManager.HelpMessage(_RF_Alert_ShipDamage_Overheat)
            DamageValueSafe(Hull, Damage)
            float OH = _RF_OverheatDuration.GetValue()
            _RF_OverheatDuration.SetValue(OH * ( 1 + Damage ) )
            _RF_GravDriveOverheatSpell.Cast(PlayerRef)
            UsingEmergencyJump = False
        EndIf
    EndIf
    Return DidDamageOccur
EndFunction

Function HandleCooldowns()
    If Running
        If _RF_InCooldownState.IsTrue(PlayerRef)
            SequentialFastJumps += 1
        Else
            SequentialFastJumps = 0
        EndIF
        QuickTrace("HandleCooldowns fired with Sequential jumps " + SequentialFastJumps )
        _rf_GravDriveDetectiveSpell.Cast(PlayerRef)
        If SequentialFastJumps > 3
            If !FuelManager.CoinFlip(25)
                _RF_CooldownDuration.SetValue(Utility.RandomInt(280, 460))
                _RF_GravDriveCooldownSpell.Cast(PlayerRef)
            EndIf
        EndIF
    EndIf
EndFunction


;-------------------------------------------------
;--------------------- EVENTS --------------------
;-------------------------------------------------


Event OnQuestInit()
    StartTimer(3)
EndEvent


Event OnTimer(int aiTimerID)

    PlayerRef = Game.GetPlayer()
    Debug.Trace("RF Travel system started dormant")

EndEvent


; If this returns true that means this jump failed explosively.
bool Function HandleTravelRestriction(Location akOldLoc, Location akNewLoc)
    bool OKGO = false
    If Running
        MyShip = FuelManager.GetShip("Travel")
        OKGO = HandleEmergencyJump()
    Else
        KillAllTravel()
    EndIF
    If !FuelManager.ModRunning()
        QuickTrace("Fuel Manager killing process with Main system disabled")
        KillAllTravel()
        Running = False
    EndIf
    Return OKGO
EndFunction


; Event received when a ship grav jump event occurs - State { Initiated = 0, AnimStarted = 1, Completed = 2, Failed = 3 }
Function HandleGravJump(Location aDestination, int aState)
    HandleCooldowns()
EndFunction



;-------------------------------------------------
;---------------- EXTERNAL CALLS -----------------
;-------------------------------------------------



; This simply checks to ensure we have attempted to conduct repairs of some kind and re-enables grav jumping if so
bool Function HandleOverheatEnd()
    _RF_OverheatDuration.SetValue(180)
    bool OK = false
    If !Running
        float HP = MyShip.GetValue(SpaceshipGravDriveHealth)
        ActorValue Hull = Game.GetHealthAV()
        float HUL = MyShip.GetValuePercentage(Hull)
        int FUL = GFLI()
        If HP >= 0 || ( HUL > 0.5 )
            float GravDriveHealthRestore = Math.abs(HP) * 2
            OK = True
            MyShip.RestoreValue(SpaceshipGravDriveHealth, GravDriveHealthRestore)
        EndIF
        QuickTrace("HandleOverheatEnd found our Grav Drive health at" + HP + " and HULL: " + HUL + " Fuel: " + FUL + " for " + OK)
    Else
        OK = true
        QuickTrace("HandleOverheatEnd killing process with Running: " + Running)
        float HP = MyShip.GetValue(SpaceshipGravDriveHealth)
        float GravDriveHealthRestore = Math.abs(HP) * 2
        MyShip.RestoreValue(SpaceshipGravDriveHealth, GravDriveHealthRestore)
    EndIF
    Return OK
EndFunction